package org.msh.tb.reports2.me;

import org.msh.reports.variables.Variable;
import org.msh.tb.application.App;
import org.msh.tb.client.shared.model.CIndicatorRequest;
import org.msh.tb.client.shared.model.CIndicatorResponse;
import org.msh.tb.entities.AdministrativeUnit;
import org.msh.tb.entities.enums.CaseClassification;
import org.msh.tb.reports2.ReportGenerator;
import org.msh.tb.reports2.ReportResources;
import org.msh.tb.reports2.variables.NotifTreatmentSiteVariable;

import javax.persistence.EntityManager;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

/**
 * Ulitity class to make it easier to generate indicators
 * Created by rmemoria on 8/12/16.
 */
public class IndicatorBuilder {

    public enum IndicatorPeriod {
        INI_TREATMENT_DATE,
        REGISTRATION_DATE
    }

    private Integer adminUnitId;
    private Integer unitId;
    private Integer year;
    private Integer quarter;
    private CaseClassification classification;
    private ArrayList<Variable> columnVariables;
    private ArrayList<Variable> rowVariables;
    private HashMap<String, String> filters;
    private IndicatorPeriod period = IndicatorPeriod.REGISTRATION_DATE;


    public IndicatorBuilder(Integer unitId, Integer adminUnitId, Integer quarter, Integer year, CaseClassification classification) {
        this.adminUnitId = adminUnitId;
        this.unitId = unitId;
        this.quarter = quarter;
        this.year = year;
        this.classification = classification;
    }

    public IndicatorBuilder addColumnVariables(String... vars) {
        if (columnVariables == null) {
            columnVariables = new ArrayList<Variable>();
        }

        ReportResources res = ReportResources.instance();

        for (String s: vars) {
            Variable var = res.findVariableById(s);
            if (var == null) {
                throw new RuntimeException("Variable '" + s + "' not found for report column");
            }
            columnVariables.add(var);
        }

        return this;
    }

    /**
     * Add variables to the row
     * @param vars
     * @return
     */
    public IndicatorBuilder addRowVariables(String... vars) {
        if (rowVariables == null) {
            rowVariables = new ArrayList<Variable>();
        }

        ReportResources res = ReportResources.instance();

        for (String s: vars) {
            Variable var = res.findVariableById(s);
            if (var == null) {
                throw new RuntimeException("Variable '" + s + "' not found for report row");
            }
            rowVariables.add(var);
        }

        return this;
    }

    public IndicatorBuilder addFilter(String filter, String value) {
        if (filters == null) {
            filters = new HashMap<String, String>();
        }

        filters.put(filter, value);

        return this;
    }

    /**
     * Clear the variables and filters in use
     */
    public IndicatorBuilder clear() {
        if (filters != null) {
            filters.clear();
        }

        if (columnVariables != null) {
            columnVariables.clear();
        }

        if (rowVariables != null) {
            rowVariables.clear();
        }

        return this;
    }

    /**
     * Generate the indicator
     * @return
     */
    public TableData generate(String title) {
        CIndicatorRequest req = new CIndicatorRequest();

        req.setColVariables(convertVars(columnVariables));
        req.setRowVariables(convertVars(rowVariables));

        req.setFilters(new HashMap<String, String>());
        if (filters != null) {
            req.getFilters().putAll(filters);
        }

        prepareRequest(req);

        CIndicatorResponse resp = ReportGenerator.generateReport(req);

        return new TableData(title,
                resp,
                "Cases",
                //generateTitle(columnVariables),
                generateTitle(rowVariables),
                rowVariables.size() > 1);
    }

    private void prepareRequest(CIndicatorRequest req) {
        if (adminUnitId != null) {
            EntityManager em = App.getEntityManager();
            AdministrativeUnit au = em.find(AdministrativeUnit.class, adminUnitId);
            // administrative unit
            req.getFilters().put(NotifTreatmentSiteVariable.FILTER_ID, au.getCode());
        } else if (unitId != null) {
            req.getFilters().put("treatunit", "0," + unitId);
        }

        if (year != null) {
            int iniMonth = quarter == null ? 1 : ((quarter - 1) * 3) + 1;
            int endMonth = quarter == null ? 12 : iniMonth + 2;

            String filterId = period == IndicatorPeriod.INI_TREATMENT_DATE ? "initreat" : "regdate";

            req.getFilters().put(filterId, "M," + iniMonth + "," + year + "," + endMonth + "," + year);
        }

        if (classification != null && !req.getFilters().containsKey("classif")) {
            req.getFilters().put("classif", Integer.toString(classification.ordinal()));
        }
    }

    private String generateTitle(List<Variable> vars) {
        String s = "";

        for (Variable var: vars) {
            if (!s.isEmpty()) {
                s += " / ";
            }
            s += var.getLabel();
        }

        return s;
    }

    private ArrayList<String> convertVars(List<Variable> vars) {
        ArrayList<String> lst = new ArrayList<String>();

        if (vars != null) {
            for (Variable var: vars) {
                lst.add(var.getId());
            }
        }

        return lst;
    }

    public IndicatorPeriod getPeriod() {
        return period;
    }

    public IndicatorBuilder setPeriod(IndicatorPeriod period) {
        this.period = period;
        return this;
    }

    public CaseClassification getClassification() {
        return classification;
    }

    public void setClassification(CaseClassification classification) {
        this.classification = classification;
    }
}
